"use client"

import { useState, useEffect } from "react"
import Link from "next/link"
import Logo from "./logo"
import { Menu, X, ChevronDown } from "lucide-react"

const navItems = [
  {
    name: "Services",
    href: "#services",
    dropdown: [
      { name: "AI Solutions", href: "/services/ai" },
      { name: "Fintech", href: "/services/fintech" },
      { name: "Consultancy", href: "/services/consultancy" },
      { name: "Technology", href: "/services/technology" },
    ],
  },
  { name: "About", href: "/about" },
  { name: "Case Studies", href: "/case-studies" },
  { name: "Blog", href: "/blog" },
  { name: "Contact", href: "/contact" },
]

export default function Navbar() {
  const [isOpen, setIsOpen] = useState(false)
  const [scrolled, setScrolled] = useState(false)
  const [activeDropdown, setActiveDropdown] = useState<string | null>(null)

  useEffect(() => {
    const handleScroll = () => {
      if (window.scrollY > 10) {
        setScrolled(true)
      } else {
        setScrolled(false)
      }
    }

    window.addEventListener("scroll", handleScroll)
    return () => window.removeEventListener("scroll", handleScroll)
  }, [])

  const toggleDropdown = (name: string) => {
    if (activeDropdown === name) {
      setActiveDropdown(null)
    } else {
      setActiveDropdown(name)
    }
  }

  return (
    <header
      className={`fixed top-0 left-0 right-0 z-50 transition-all duration-300 ${
        scrolled ? "bg-background/80 backdrop-blur-md shadow-md py-2" : "bg-transparent py-4"
      }`}
    >
      <div className="container mx-auto px-4 flex items-center justify-between">
        <Link href="/" className="z-50">
          <Logo size="md" />
        </Link>

        {/* Desktop Navigation */}
        <nav className="hidden md:flex items-center space-x-8">
          {navItems.map((item) => (
            <div key={item.name} className="relative group">
              {item.dropdown ? (
                <button
                  className="flex items-center text-foreground/80 hover:text-primary transition-colors"
                  onClick={() => toggleDropdown(item.name)}
                >
                  {item.name}
                  <ChevronDown className="ml-1 h-4 w-4" />
                </button>
              ) : (
                <Link href={item.href} className="text-foreground/80 hover:text-primary transition-colors">
                  {item.name}
                </Link>
              )}

              {item.dropdown && (
                <div className="absolute left-0 mt-2 w-48 rounded-md shadow-lg bg-background border border-border overflow-hidden opacity-0 invisible group-hover:opacity-100 group-hover:visible transition-all duration-300">
                  <div className="py-1">
                    {item.dropdown.map((subItem) => (
                      <Link
                        key={subItem.name}
                        href={subItem.href}
                        className="block px-4 py-2 text-sm text-foreground hover:bg-muted transition-colors"
                      >
                        {subItem.name}
                      </Link>
                    ))}
                  </div>
                </div>
              )}
            </div>
          ))}

          <Link
            href="/contact"
            className="px-5 py-2 rounded-full bg-gradient-to-r from-primary via-secondary to-accent text-white font-medium hover:shadow-lg transition-all"
          >
            Get Started
          </Link>
        </nav>

        {/* Mobile Navigation Toggle */}
        <button className="md:hidden z-50" onClick={() => setIsOpen(!isOpen)}>
          {isOpen ? <X className="h-6 w-6 text-foreground" /> : <Menu className="h-6 w-6 text-foreground" />}
        </button>

        {/* Mobile Navigation Menu */}
        <div
          className={`fixed inset-0 bg-background/95 backdrop-blur-sm flex flex-col justify-center items-center transition-all duration-300 ${
            isOpen ? "opacity-100 visible" : "opacity-0 invisible"
          } md:hidden`}
        >
          <nav className="flex flex-col items-center space-y-6 w-full px-8">
            {navItems.map((item) => (
              <div key={item.name} className="w-full">
                {item.dropdown ? (
                  <div className="w-full">
                    <button
                      className="flex items-center justify-between w-full text-lg font-medium py-2 border-b border-border"
                      onClick={() => toggleDropdown(item.name)}
                    >
                      {item.name}
                      <ChevronDown
                        className={`h-5 w-5 transition-transform ${activeDropdown === item.name ? "rotate-180" : ""}`}
                      />
                    </button>

                    {activeDropdown === item.name && (
                      <div className="mt-2 pl-4 space-y-2">
                        {item.dropdown.map((subItem) => (
                          <Link
                            key={subItem.name}
                            href={subItem.href}
                            className="block py-2 text-foreground/80 hover:text-primary"
                            onClick={() => setIsOpen(false)}
                          >
                            {subItem.name}
                          </Link>
                        ))}
                      </div>
                    )}
                  </div>
                ) : (
                  <Link
                    href={item.href}
                    className="block text-lg font-medium py-2 border-b border-border w-full"
                    onClick={() => setIsOpen(false)}
                  >
                    {item.name}
                  </Link>
                )}
              </div>
            ))}

            <Link
              href="/contact"
              className="w-full px-5 py-3 rounded-full bg-gradient-to-r from-primary via-secondary to-accent text-white font-medium text-center"
              onClick={() => setIsOpen(false)}
            >
              Get Started
            </Link>
          </nav>
        </div>
      </div>
    </header>
  )
}

