"use client"

import { useState, useEffect, useRef } from "react"
import Image from "next/image"
import Link from "next/link"
import { motion, useScroll, useTransform } from "framer-motion"
import { ArrowRight, Brain, LineChart, Users, Code, ChevronRight, CheckCircle2 } from "lucide-react"

export default function Home() {
  const [isVisible, setIsVisible] = useState(false)
  const heroRef = useRef<HTMLDivElement>(null)
  const { scrollYProgress } = useScroll()
  const y = useTransform(scrollYProgress, [0, 1], ["0%", "50%"])

  useEffect(() => {
    setIsVisible(true)

    const observer = new IntersectionObserver(
      ([entry]) => {
        if (entry.isIntersecting) {
          setIsVisible(true)
          observer.unobserve(entry.target)
        }
      },
      { threshold: 0.1 },
    )

    if (heroRef.current) {
      observer.observe(heroRef.current)
    }

    return () => {
      if (heroRef.current) {
        observer.unobserve(heroRef.current)
      }
    }
  }, [])

  const fadeIn = {
    hidden: { opacity: 0, y: 20 },
    visible: {
      opacity: 1,
      y: 0,
      transition: { duration: 0.6 },
    },
  }

  const staggerContainer = {
    hidden: { opacity: 0 },
    visible: {
      opacity: 1,
      transition: {
        staggerChildren: 0.2,
      },
    },
  }

  return (
    <>
      {/* Hero Section */}
      <section ref={heroRef} className="relative min-h-screen flex items-center justify-center overflow-hidden pt-20">
        <div className="absolute inset-0 bg-gradient-to-b from-background via-background to-background/50 z-10"></div>

        {/* Dynamic Background Animation */}
        <div className="absolute inset-0 z-0">
          <motion.div
            className="absolute inset-0 opacity-30"
            style={{
              backgroundImage: "url('/placeholder.svg?height=1080&width=1920&text=Dynamic+Background')",
              backgroundSize: "cover",
              backgroundPosition: "center",
              y,
            }}
          />
          <div className="absolute top-0 left-0 w-full h-full bg-[radial-gradient(circle_at_30%_20%,rgba(67,56,202,0.15),transparent_40%)]"></div>
          <div className="absolute bottom-0 right-0 w-full h-full bg-[radial-gradient(circle_at_70%_80%,rgba(147,51,234,0.15),transparent_40%)]"></div>
        </div>

        <div className="container mx-auto px-4 relative z-20">
          <motion.div
            className="max-w-4xl mx-auto text-center"
            initial="hidden"
            animate={isVisible ? "visible" : "hidden"}
            variants={staggerContainer}
          >
            <motion.h1 className="text-4xl md:text-6xl lg:text-7xl font-bold mb-6" variants={fadeIn}>
              <span className="gradient-text">Pioneering</span> the Future of Technology
            </motion.h1>

            <motion.p className="text-xl md:text-2xl text-foreground/80 mb-8 max-w-3xl mx-auto" variants={fadeIn}>
              Alpha3 Solution delivers cutting-edge AI, Fintech, Consultancy, and Technology services to transform your
              business.
            </motion.p>

            <motion.div className="flex flex-col sm:flex-row items-center justify-center gap-4" variants={fadeIn}>
              <Link
                href="/contact"
                className="px-8 py-3 rounded-full bg-gradient-to-r from-primary via-secondary to-accent text-white font-medium hover:shadow-lg transition-all text-lg group"
              >
                Get Started
                <motion.span
                  className="inline-block ml-2"
                  initial={{ x: 0 }}
                  whileHover={{ x: 5 }}
                  transition={{ type: "spring", stiffness: 300 }}
                >
                  <ArrowRight className="h-5 w-5" />
                </motion.span>
              </Link>

              <Link
                href="/services"
                className="px-8 py-3 rounded-full border border-border bg-background/50 backdrop-blur-sm hover:bg-background/80 transition-all text-lg flex items-center group"
              >
                Explore Services
                <motion.span
                  className="inline-block ml-2"
                  initial={{ x: 0 }}
                  whileHover={{ x: 5 }}
                  transition={{ type: "spring", stiffness: 300 }}
                >
                  <ArrowRight className="h-5 w-5" />
                </motion.span>
              </Link>
            </motion.div>
          </motion.div>
        </div>

        {/* Floating Elements */}
        <div className="absolute bottom-0 left-0 right-0 h-64 z-10 pointer-events-none">
          <div className="relative h-full max-w-7xl mx-auto">
            <motion.div
              className="absolute bottom-20 left-[10%] w-16 h-16 rounded-xl bg-primary/20 backdrop-blur-md"
              animate={{ y: [0, -20, 0] }}
              transition={{ repeat: Number.POSITIVE_INFINITY, duration: 5, ease: "easeInOut" }}
            ></motion.div>
            <motion.div
              className="absolute bottom-40 left-[30%] w-24 h-24 rounded-full bg-secondary/20 backdrop-blur-md"
              animate={{ y: [0, -30, 0] }}
              transition={{ repeat: Number.POSITIVE_INFINITY, duration: 7, ease: "easeInOut", delay: 1 }}
            ></motion.div>
            <motion.div
              className="absolute bottom-10 left-[60%] w-20 h-20 rounded-lg bg-accent/20 backdrop-blur-md"
              animate={{ y: [0, -25, 0] }}
              transition={{ repeat: Number.POSITIVE_INFINITY, duration: 6, ease: "easeInOut", delay: 2 }}
            ></motion.div>
            <motion.div
              className="absolute bottom-30 left-[80%] w-12 h-12 rounded-full bg-primary/20 backdrop-blur-md"
              animate={{ y: [0, -15, 0] }}
              transition={{ repeat: Number.POSITIVE_INFINITY, duration: 4, ease: "easeInOut", delay: 1.5 }}
            ></motion.div>
          </div>
        </div>

        <motion.div
          className="absolute bottom-10 left-1/2 transform -translate-x-1/2 z-20"
          animate={{ y: [0, 10, 0] }}
          transition={{ repeat: Number.POSITIVE_INFINITY, duration: 2, ease: "easeInOut" }}
        >
          <ChevronRight className="h-8 w-8 rotate-90 text-foreground/50" />
        </motion.div>
      </section>

      {/* Services Section */}
      <section id="services" className="py-20 bg-gradient-to-b from-background to-background/95">
        <div className="container mx-auto px-4">
          <div className="text-center max-w-3xl mx-auto mb-16">
            <h2 className="text-3xl md:text-4xl font-bold mb-4">Our Expertise</h2>
            <p className="text-lg text-foreground/80">
              Comprehensive solutions tailored to meet the evolving needs of modern businesses
            </p>
          </div>

          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-8">
            {/* AI Solutions */}
            <motion.div
              className="bg-background/50 backdrop-blur-sm border border-border rounded-xl p-6 hover:shadow-lg transition-all group"
              whileHover={{ y: -10 }}
              initial={{ opacity: 0, y: 20 }}
              whileInView={{ opacity: 1, y: 0 }}
              viewport={{ once: true }}
              transition={{ duration: 0.5 }}
            >
              <div className="w-14 h-14 rounded-lg bg-primary/10 flex items-center justify-center mb-6 group-hover:bg-primary/20 transition-colors">
                <Brain className="h-7 w-7 text-primary" />
              </div>
              <h3 className="text-xl font-semibold mb-3">AI Solutions</h3>
              <p className="text-foreground/70 mb-4">
                Harness the power of artificial intelligence to automate processes, gain insights, and drive innovation.
              </p>
              <Link
                href="/services/ai"
                className="inline-flex items-center text-primary hover:text-primary/80 transition-colors group"
              >
                Learn more
                <motion.span
                  className="inline-block ml-2"
                  initial={{ x: 0 }}
                  whileHover={{ x: 5 }}
                  transition={{ type: "spring", stiffness: 300 }}
                >
                  <ArrowRight className="h-4 w-4" />
                </motion.span>
              </Link>
            </motion.div>

            {/* Fintech */}
            <motion.div
              className="bg-background/50 backdrop-blur-sm border border-border rounded-xl p-6 hover:shadow-lg transition-all group"
              whileHover={{ y: -10 }}
              initial={{ opacity: 0, y: 20 }}
              whileInView={{ opacity: 1, y: 0 }}
              viewport={{ once: true }}
              transition={{ duration: 0.5 }}
            >
              <div className="w-14 h-14 rounded-lg bg-secondary/10 flex items-center justify-center mb-6 group-hover:bg-secondary/20 transition-colors">
                <LineChart className="h-7 w-7 text-secondary" />
              </div>
              <h3 className="text-xl font-semibold mb-3">Fintech Services</h3>
              <p className="text-foreground/70 mb-4">
                Revolutionize financial operations with cutting-edge technology solutions designed for the digital
                economy.
              </p>
              <Link
                href="/services/fintech"
                className="inline-flex items-center text-secondary hover:text-secondary/80 transition-colors group"
              >
                Learn more
                <motion.span
                  className="inline-block ml-2"
                  initial={{ x: 0 }}
                  whileHover={{ x: 5 }}
                  transition={{ type: "spring", stiffness: 300 }}
                >
                  <ArrowRight className="h-4 w-4" />
                </motion.span>
              </Link>
            </motion.div>

            {/* Consultancy */}
            <motion.div
              className="bg-background/50 backdrop-blur-sm border border-border rounded-xl p-6 hover:shadow-lg transition-all group"
              whileHover={{ y: -10 }}
              initial={{ opacity: 0, y: 20 }}
              whileInView={{ opacity: 1, y: 0 }}
              viewport={{ once: true }}
              transition={{ duration: 0.5 }}
            >
              <div className="w-14 h-14 rounded-lg bg-accent/10 flex items-center justify-center mb-6 group-hover:bg-accent/20 transition-colors">
                <Users className="h-7 w-7 text-accent" />
              </div>
              <h3 className="text-xl font-semibold mb-3">Consultancy</h3>
              <p className="text-foreground/70 mb-4">
                Strategic guidance from industry experts to help navigate complex challenges and drive sustainable
                growth.
              </p>
              <Link
                href="/services/consultancy"
                className="inline-flex items-center text-accent hover:text-accent/80 transition-colors group"
              >
                Learn more
                <motion.span
                  className="inline-block ml-2"
                  initial={{ x: 0 }}
                  whileHover={{ x: 5 }}
                  transition={{ type: "spring", stiffness: 300 }}
                >
                  <ArrowRight className="h-4 w-4" />
                </motion.span>
              </Link>
            </motion.div>

            {/* Technology Services */}
            <motion.div
              className="bg-background/50 backdrop-blur-sm border border-border rounded-xl p-6 hover:shadow-lg transition-all group"
              whileHover={{ y: -10 }}
              initial={{ opacity: 0, y: 20 }}
              whileInView={{ opacity: 1, y: 0 }}
              viewport={{ once: true }}
              transition={{ duration: 0.5 }}
            >
              <div className="w-14 h-14 rounded-lg bg-primary/10 flex items-center justify-center mb-6 group-hover:bg-primary/20 transition-colors">
                <Code className="h-7 w-7 text-primary" />
              </div>
              <h3 className="text-xl font-semibold mb-3">Technology Services</h3>
              <p className="text-foreground/70 mb-4">
                End-to-end technology solutions that empower businesses to stay ahead in the digital transformation
                journey.
              </p>
              <Link
                href="/services/technology"
                className="inline-flex items-center text-primary hover:text-primary/80 transition-colors group"
              >
                Learn more
                <motion.span
                  className="inline-block ml-2"
                  initial={{ x: 0 }}
                  whileHover={{ x: 5 }}
                  transition={{ type: "spring", stiffness: 300 }}
                >
                  <ArrowRight className="h-4 w-4" />
                </motion.span>
              </Link>
            </motion.div>
          </div>
        </div>
      </section>

      {/* About Section */}
      <section className="py-20 bg-gradient-to-b from-background/95 to-background">
        <div className="container mx-auto px-4">
          <div className="grid grid-cols-1 lg:grid-cols-2 gap-12 items-center">
            <motion.div
              initial={{ opacity: 0, x: -50 }}
              whileInView={{ opacity: 1, x: 0 }}
              viewport={{ once: true }}
              transition={{ duration: 0.6 }}
            >
              <h2 className="text-3xl md:text-4xl font-bold mb-6">Why Choose Alpha3 Solution?</h2>
              <p className="text-lg text-foreground/80 mb-8">
                At Alpha3 Solution, we combine cutting-edge technology with deep industry expertise to deliver
                transformative solutions that drive real business outcomes.
              </p>

              <div className="space-y-4">
                <motion.div
                  className="flex items-start"
                  initial={{ opacity: 0, y: 20 }}
                  whileInView={{ opacity: 1, y: 0 }}
                  viewport={{ once: true }}
                  transition={{ duration: 0.5 }}
                >
                  <CheckCircle2 className="h-6 w-6 text-primary mr-3 mt-0.5 flex-shrink-0" />
                  <div>
                    <h3 className="font-medium text-lg">Innovative Approach</h3>
                    <p className="text-foreground/70">
                      We stay ahead of technological trends to provide forward-thinking solutions.
                    </p>
                  </div>
                </motion.div>

                <motion.div
                  className="flex items-start"
                  initial={{ opacity: 0, y: 20 }}
                  whileInView={{ opacity: 1, y: 0 }}
                  viewport={{ once: true }}
                  transition={{ duration: 0.5, delay: 0.1 }}
                >
                  <CheckCircle2 className="h-6 w-6 text-primary mr-3 mt-0.5 flex-shrink-0" />
                  <div>
                    <h3 className="font-medium text-lg">Expert Team</h3>
                    <p className="text-foreground/70">
                      Our specialists bring years of experience across multiple industries.
                    </p>
                  </div>
                </motion.div>

                <motion.div
                  className="flex items-start"
                  initial={{ opacity: 0, y: 20 }}
                  whileInView={{ opacity: 1, y: 0 }}
                  viewport={{ once: true }}
                  transition={{ duration: 0.5, delay: 0.2 }}
                >
                  <CheckCircle2 className="h-6 w-6 text-primary mr-3 mt-0.5 flex-shrink-0" />
                  <div>
                    <h3 className="font-medium text-lg">Tailored Solutions</h3>
                    <p className="text-foreground/70">
                      We customize our services to meet your specific business needs.
                    </p>
                  </div>
                </motion.div>

                <motion.div
                  className="flex items-start"
                  initial={{ opacity: 0, y: 20 }}
                  whileInView={{ opacity: 1, y: 0 }}
                  viewport={{ once: true }}
                  transition={{ duration: 0.5, delay: 0.3 }}
                >
                  <CheckCircle2 className="h-6 w-6 text-primary mr-3 mt-0.5 flex-shrink-0" />
                  <div>
                    <h3 className="font-medium text-lg">Proven Results</h3>
                    <p className="text-foreground/70">
                      Our track record speaks for itself with successful implementations worldwide.
                    </p>
                  </div>
                </motion.div>
              </div>

              <div className="mt-8">
                <Link
                  href="/about"
                  className="px-6 py-2.5 rounded-full border border-primary text-primary hover:bg-primary hover:text-white transition-all inline-flex items-center group"
                >
                  Learn more about us
                  <motion.span
                    className="inline-block ml-2"
                    initial={{ x: 0 }}
                    whileHover={{ x: 5 }}
                    transition={{ type: "spring", stiffness: 300 }}
                  >
                    <ArrowRight className="h-4 w-4" />
                  </motion.span>
                </Link>
              </div>
            </motion.div>

            <motion.div
              className="relative"
              initial={{ opacity: 0, x: 50 }}
              whileInView={{ opacity: 1, x: 0 }}
              viewport={{ once: true }}
              transition={{ duration: 0.6 }}
            >
              <div className="relative z-10 rounded-2xl overflow-hidden shadow-xl">
                <Image
                  src="/placeholder.svg?height=600&width=800"
                  alt="Alpha3 Solution Team"
                  width={800}
                  height={600}
                  className="w-full h-auto object-cover"
                />
              </div>

              <div className="absolute -bottom-6 -right-6 w-64 h-64 bg-gradient-to-br from-primary to-accent rounded-full opacity-20 blur-3xl -z-10"></div>
              <div className="absolute -top-6 -left-6 w-64 h-64 bg-gradient-to-br from-secondary to-primary rounded-full opacity-20 blur-3xl -z-10"></div>
            </motion.div>
          </div>
        </div>
      </section>

      {/* CTA Section */}
      <section className="py-20 relative overflow-hidden">
        <div className="absolute inset-0 bg-gradient-to-r from-primary/20 via-secondary/20 to-accent/20"></div>
        <div className="container mx-auto px-4 relative z-10">
          <div className="max-w-4xl mx-auto text-center">
            <motion.h2
              className="text-3xl md:text-4xl font-bold mb-6"
              initial={{ opacity: 0, y: 20 }}
              whileInView={{ opacity: 1, y: 0 }}
              viewport={{ once: true }}
              transition={{ duration: 0.5 }}
            >
              Ready to Transform Your Business?
            </motion.h2>

            <motion.p
              className="text-lg text-foreground/80 mb-8"
              initial={{ opacity: 0, y: 20 }}
              whileInView={{ opacity: 1, y: 0 }}
              viewport={{ once: true }}
              transition={{ duration: 0.5, delay: 0.1 }}
            >
              Partner with Alpha3 Solution to leverage cutting-edge technology and expert consultancy for your business
              growth.
            </motion.p>

            <motion.div
              initial={{ opacity: 0, y: 20 }}
              whileInView={{ opacity: 1, y: 0 }}
              viewport={{ once: true }}
              transition={{ duration: 0.5, delay: 0.2 }}
            >
              <Link
                href="/contact"
                className="px-8 py-3 rounded-full bg-gradient-to-r from-primary via-secondary to-accent text-white font-medium hover:shadow-lg transition-all text-lg inline-block group"
              >
                Schedule a Consultation
                <motion.span
                  className="inline-block ml-2"
                  initial={{ x: 0 }}
                  whileHover={{ x: 5 }}
                  transition={{ type: "spring", stiffness: 300 }}
                >
                  <ArrowRight className="h-5 w-5" />
                </motion.span>
              </Link>
            </motion.div>
          </div>
        </div>
      </section>
    </>
  )
}

